<?php
class TheodoreThemeSettingsAPI {
	
	private $slug;
	private $pageTitle;
	private $menuTitle;
	private $error = array();
	private $sections = array();
	private $fields = array();
	
	public function __construct () {}
	
	public function execute () {
		
		if ( $this->slug == '' ) {
			$this->error[] = 'Slug is not defined';
		}
		
		if ( $this->pageTitle == '' ) {
			$this->error[] = 'Page Title is not defined';
		}
		
		if ( $this->menuTitle == '' ) {
			$this->error[] = 'Menu Title is not defined';
		}
		
		if ( count( $this->error ) < 1 ) {
			add_action( 'admin_init', array( $this, 'adminInit' ) );
			add_action( 'admin_menu', array( $this, 'adminMenu' ) );
		}
		else {
			echo 'Cannot create settings page: <ul>';
			foreach ( $this->error as $v ) {
				echo '<li>' . esc_html( $v ) . '</li>';
			}
			echo '</ul>';
		}
	}
	
	public function setSlug ( $slug ) {
		$this->slug = $slug;
	}
	
	public function setPageTitle ( $page_title ) {
		$this->pageTitle = $page_title;
	}
	
	public function setMenuTitle ( $menu_title ) {
		$this->menuTitle = $menu_title;
	}
	
	public function setData ( $data ) {
		foreach ( $data as $k => $section ) {
			
			/* Add to array of sections */
			$this->sections[] = array(
				'id' => $k,
				'title' => isset( $section['title'] ) ? $section['title'] : '',
				'description' => isset( $section['description'] ) ? $section['description'] : ''
			);
			
			/* 	Jump to next item if no fields item. The section will still be 
				rendered but it won't have any field. */
			if ( ! isset( $section['fields'] )  || ! is_array( $section['fields'] ) ) {
				continue;
			}
			
			/* Add to array of fields */
			foreach ( $section['fields'] as $field ) {
				$field['section'] = $k;
				$this->fields[] = $field;
			}
		}
	}
	
	public function adminInit() {
		
		foreach ( $this->sections as $section ) {
			
			$id = isset( $section['id'] ) ? $section['id'] : '';
			$title = isset( $section['title'] ) ? $section['title'] : '';
			
			if ( isset( $section['description'] ) && ! empty( $section['description'] ) ) {
				$description = '<div class="inside">' . $section['description'] . '</div>';
				$callback = create_function( '', 'echo "' . str_replace( '"', '\"', $description ) . '";' );
			}
			else {
				$callback = '__return_false';
			}
			
			add_settings_section( $id, $title, $callback, $this->slug );
			register_setting( $this->slug, $id, array( $this, 'sanitizeOptions' ) );
		}
		
		foreach ( $this->fields as $k => $field ) {
			
			$id = isset( $field['id'] ) ? $field['id'] : '';
			$title = isset( $field['label'] ) ? $field['label'] : '';
			$section = isset( $field['section'] ) ? $field['section'] : '';
			$type = isset( $field['type'] ) ? $field['type'] : 'text';
			
			$args = array(
				'id' => $id,
				'description' => isset( $field['description'] ) ? $field['description'] : '',
				'name' => $title,
				'section' => $section,
				'maxlength' => isset( $field['maxlength'] ) ? $field['maxlength'] : null,
				'size' => isset( $field['size'] ) ? $field['size'] : null,
				'classes' => isset( $field['classes'] ) ? $field['classes'] : null,
				'options' => isset( $field['options'] ) ? $field['options'] : '',
				'std' => isset( $field['default'] ) ? $field['default'] : '',
				'sanitize_callback' => isset( $field['sanitize_callback'] ) ? $field['sanitize_callback'] : ''
			);
			
			add_settings_field( $section . '[' . $id . ']', $title, array( $this, 'callback_' . $type ), $this->slug, $section, $args );
		}
		
	}
	
	public function adminMenu() {
		add_theme_page( $this->pageTitle, $this->menuTitle, 'manage_options', $this->slug, array( $this, 'output' ) );
	}
	
	/**
	 * Display a hidden field for settings field. This is useful for setting an option without ui.
	 * In the case of Theodore Theme, this field is used for storing css style which is generated after
	 * Wordpress save the settings options.
	 *
	 * @param array $args settings field args
	 */
	public function callback_hidden ( $args ) {
		$value = $this->getOption( $args['id'], $args['section'], $args['std'] );
		$size = isset( $args['size'] ) && !is_null( $args['size'] ) ? $args['size'] : 'regular';
		$classes = isset( $args['classes'] ) ? $args['classes'] : '';
		
		$html = sprintf( '<input type="hidden" id="%1$s_%2$s" name="%1$s[%2$s]" value="%3$s"/> ', $args['section'], $args['id'], $value );
        
		echo $html;
	}
	
	/**
	 * Display a text field for settings field
	 *
	 * @param array $args settings field args
	 */
	public function callback_media_url ( $args ) {
		$value = esc_url( $this->getOption( $args['id'], $args['section'], $args['std'] ) );
		$meta_value = $this->getOption( $args['id'] . '_meta', $args['section'], $args['std'] );
		$size = isset( $args['size'] ) && !is_null( $args['size'] ) ? $args['size'] : 'regular';
		$classes = isset( $args['classes'] ) ? $args['classes'] : '';
		
		$html = sprintf( '<input type="url" class="%1$s-text %5$s" id="%2$s_%3$s" name="%2$s[%3$s]" value="%4$s"/> ', $size, $args['section'], $args['id'], $value, $classes );
        $html.= sprintf( '<input type="hidden" id="%2$s_%3$s" name="%2$s[%3$s]" value=\'%4$s\'/> ', $size, $args['section'], $args['id'] . '_meta', $meta_value, $classes );
		/* TRANSLATORS: Button label to remove the already chosen image on the Theodore Settings page. (Back end) */
		$html.= sprintf( '<button class="uploader_button button" data-clear-text="' . esc_html__( 'Remove Image', 'theodore' ) . '" data-target="%1$s_%2$s" data-meta="%1$s_%3$s" data-preview="preview_%1$s_%2$s">' . 
		/* TRANSLATORS: Button label to choose an image on the Theodore Settings page. (Back end) */
		esc_html__( 'Choose Image', 'theodore' ) . '</button>', $args['section'], $args['id'], $args['id'] . '_meta' );
		$html.= sprintf( '<span id="preview_%1$s_%2$s" class="image-preview">' . ( $value != '' ? '<img src="%3$s" />' : '' ) . '</span>', $args['section'], $args['id'], $value );
		$html.= sprintf( '<p class="description"> %s</p>', $args['description'] );
		
		echo $html;
	}
	
	/**
	 * Display a text field for settings field
	 *
	 * @param array $args settings field args
	 */
	public function callback_text ( $args ) {
		$value = esc_attr( $this->getOption( $args['id'], $args['section'], $args['std'] ) );
		$size = isset( $args['size'] ) && !is_null( $args['size'] ) ? $args['size'] : 'regular';
		$classes = isset( $args['classes'] ) ? $args['classes'] : '';
		
		$html = sprintf( '<input type="text" class="%1$s-text %5$s" id="%2$s[%3$s]" name="%2$s[%3$s]" maxlength="%6$s" size="%7$s" value="%4$s"/>', $size, $args['section'], $args['id'], $value, $classes, $args['maxlength'], $args['size'] );
        $html.= sprintf( '<br><p class="description"> %s</p>', $args['description'] );
		
		echo $html;
	}
	
	/**
	 * Display a text field for settings field
	 *
	 * @param array $args settings field args
	 */
	public function callback_color ( $args ) {
		$value = esc_attr( $this->getOption( $args['id'], $args['section'], $args['std'] ) );
		$classes = isset( $args['classes'] ) ? $args['classes'] : '';
		
		$html = sprintf( '<input type="color" class="%4$s" id="%1$s[%3$s]" name="%1$s[%2$s]" value="%3$s"/>', $args['section'], $args['id'], $value, $classes );
        $html.= sprintf( '<br><p class="description"> %s</p>', $args['description'] );
		
		echo $html;
	}
	
	/**
	* Displays a checkbox for a settings field
	*
	* @param array $args settings field args
	*/
    public function callback_checkbox ( $args ) {

        $value = esc_attr( $this->getOption( $args['id'], $args['section'], $args['std'] ) );

        $html = sprintf( '<input type="hidden" name="%1$s[%2$s]" value="off" />', $args['section'], $args['id'] );
        $html.= sprintf( '<input type="checkbox" class="checkbox" id="%1$s[%2$s]" name="%1$s[%2$s]" value="on"%4$s />', $args['section'], $args['id'], $value, checked( $value, 'on', false ) );
        $html.= sprintf( '<label for="%1$s[%2$s]"> %3$s</label>', $args['section'], $args['id'], $args['description'] );

        echo $html;
    }
	
	/**
	* Displays a multicheckbox for settings field
	*
	* @param array $args settings field args
	*/
    public function callback_multicheck ( $args ) {

        $value = $this->getOption( $args['id'], $args['section'], $args['std'] );

        $html = '';
        foreach ( $args['options'] as $key => $label ) {
            $checked = isset( $value[$key] ) ? $value[$key] : '0';
            $html.= sprintf( '<input type="checkbox" class="checkbox" id="%1$s[%2$s][%3$s]" name="%1$s[%2$s][%3$s]" value="%3$s"%4$s />', $args['section'], $args['id'], $key, checked( $checked, $key, false ) );
            $html.= sprintf( '<label for="%1$s[%2$s][%4$s]"> %3$s</label><br>', $args['section'], $args['id'], $label, $key );
        }
        $html.= sprintf( '<span class="description"> %s</label>', $args['description'] );

        echo $html;
    }
	
	/**
	* Displays radio buttons for settings field
	*
	* @param array $args settings field args
	*/
    public function callback_radio ( $args ) {

        $value = $this->getOption( $args['id'], $args['section'], $args['std'] );

        $html = '';
        foreach ( $args['options'] as $key => $label ) {
            $html.= sprintf( '<input type="radio" class="radio" id="%1$s[%2$s][%3$s]" name="%1$s[%2$s]" value="%3$s"%4$s />', $args['section'], $args['id'], $key, checked( $value, $key, false ) );
            $html.= sprintf( '<label for="%1$s[%2$s][%4$s]"> %3$s</label><br>', $args['section'], $args['id'], $label, $key );
        }
        $html.= sprintf( '<span class="description"> %s</label>', $args['description'] );

        echo $html;
    }
	
	/**
	* Displays a selectbox for a settings field
	*
	* @param array $args settings field args
	*/
    public function callback_select ( $args ) {

        $value = esc_attr( $this->getOption( $args['id'], $args['section'], $args['std'] ) );
        $classes = isset( $args['classes'] ) && !is_null( $args['classes'] ) ? $args['classes'] : '';
		$size = isset( $args['size'] ) && !is_null( $args['size'] ) ? $args['size'] : 'regular';

        $html = sprintf( '<select class="%1$s %4$s" name="%2$s[%3$s]" id="%2$s[%3$s]">', $size, $args['section'], $args['id'], $classes );
        
		foreach ( $args['options'] as $key => $label ) {
            $html.= sprintf( '<option value="%s"%s>%s</option>', $key, selected( $value, $key, false ), $label );
        }
        $html.= sprintf( '</select>' );
        $html.= sprintf( '<br><p class="description"> %s</p>', $args['description'] );

        echo $html;
    }
	
	/**
	* Displays a textarea for a settings field
	*
	* @param array $args settings field args
	*/
    public function callback_textarea ( $args ) {

        $value = esc_textarea( $this->getOption( $args['id'], $args['section'], $args['std'] ) );
        $size = isset( $args['size'] ) && !is_null( $args['size'] ) ? $args['size'] : 'regular';
		$classes = isset( $args['classes'] ) ? $args['classes'] : '';

        $html = sprintf( '<textarea rows="5" cols="55" class="%1$s-text %5$s" id="%2$s[%3$s]" name="%2$s[%3$s]">%4$s</textarea>', $size, $args['section'], $args['id'], $value, $classes );
        $html.= sprintf( '<br><br><p class="description"> %s</p>', $args['description'] );

        echo $html;
    }
	
	/**
	* Displays a code editor for a settings field
	*
	* @param array $args settings field args
	*/
    public function callback_code_editor ( $args ) {

        $value = $this->getOption( $args['id'], $args['section'], $args['std'] );
        $size = isset( $args['size'] ) && !is_null( $args['size'] ) ? $args['size'] : 'regular';
		$classes = isset( $args['classes'] ) ? $args['classes'] : '';

        $html = sprintf( '<textarea rows="5" cols="55" class="%1$s-text %5$s code-editor" id="%2$s[%3$s]" name="%2$s[%3$s]" data-target="%2$s-%3$s">%4$s</textarea>', $size, $args['section'], $args['id'], $value, $classes );
		//$html.= sprintf( '<br><br><p class="description"> %s</p>', $args['description'] );

        echo $html;
    }
	
	/**
	* Displays description text
	*
	* @param array $args settings field args
	*/
    public function callback_html ( $args ) {
        echo $args['description'];
    }
	
	/**
	* Displays a rich text textarea for a settings field
	*
	* @param array $args settings field args
	*/
    public function callback_wysiwyg ( $args ) {
        /* 	When saving, the value should be altered using htmlentities so 
			that it can be saved into database. Without it, WordPress will 
			strip the tags. */
		$value = $this->getOption( $args['id'], $args['section'], $args['std'] );
        wp_editor( html_entity_decode( $value ), $args['section'] . '_' . $args['id'], array( 'textarea_rows' => 10, 'textarea_name' => $args['section'] . '[' . $args['id'] . ']' ) );		
        echo sprintf( '<br><br><p class="description"> %s</p>', $args['description'] );
    }
	
	/**
	* Sanitize callback for Settings API
	*/
    public function sanitizeOptions ( $options ) {
        foreach ( $options as $option_slug => $option_value ) {
            $sanitize_callback = $this->getSanitizeCallback( $option_slug );

            /* If callback is set, call it */
            if ( $sanitize_callback ) {
                $options[$option_slug] = call_user_func( $sanitize_callback, $option_value );
                continue;
            }

            /* Treat everything that's not an array as a string */
            if ( !is_array( $option_value ) ) {
                $options[$option_slug] = sanitize_text_field( $option_value );
                continue;
            }
        }
        return $options;
    }
	
	/**
	* Get sanitization callback for given option slug
	*
	* @param string $slug option slug
	*
	* @return mixed string or bool false
	*/
    public function getSanitizeCallback ( $slug = '' ) {
        if ( empty( $slug ) ) {
			return false;
		}
		
        /* Iterate over registered fields and see if we can find proper callback */
        foreach ( $this->fields as $section => $options ) {
			
			if ( $options['id'] != $slug ) {
				continue;
			}
			
			/* Return the callback name */
			return isset( $option['sanitize_callback']) && is_callable( $option['sanitize_callback'] ) ? $option['sanitize_callback'] : false;
		}
        
        return false;
    }
	
	/**
	* Get the value of a settings field
	*
	* @param string $option settings field name
	* @param string $section the section name this field belongs to
	* @param string $default default text if it's not found
	* @return string
	*/
    public function getOption ( $option, $section, $default = '' ) {

        $options = get_option( $section );

        if ( isset( $options[$option] ) ) {
            return $options[$option];
        }
        return $default;
    }
	
	public function output () {
		?>
		<div class="wrap">
			<h1><?php echo esc_html( $this->pageTitle ); ?></h1>
			<form method="post" action="options.php">
				
				<?php do_action( 'wsa_form_top_' . $this->slug ); ?>
				<?php do_settings_sections( $this->slug ); ?>
				<?php settings_fields( $this->slug ); ?>
				<?php do_action( 'wsa_form_bottom_' . $this->slug ); ?>
					
				<div style="padding-left: 10px">
					<?php submit_button(); ?>
				</div>
				
			</form>
		</div>
		<?php
	}
	
	
	
}


?>