<?php

/**
 * Shortcode Styling.
 *
 * This class steps in to generate styles right after WordPress saved the content.
 * The purpose of this class is to avoid inline css on the front end when the content has shortcodes
 * with styles attached to it. No need the $content, just the $attr
 */

class TheodorePluginShortcodeStyle {
	
	public static $css = '';
	private static $newRowID = 0;
	private static $newColumnID = 0;
	private static $newSocialIconLinksID = 0;
	public static $newContentSliderItemID = 0;
	public static $newPostContentSliderID = 0;
	public static $newButtonID = 0;
	
	/**
	 * Hook the function to update the 'theodore_page_style' post meta during
	 * 'save_post' action. This should be done on post creation.
	 */	
	public function __construct () {
		add_action( 'save_post', array( $this, 'savePost' ), 10, 3 );
	}
	
	public function parseShortcodes( $regex, $content, $existing = array() ) {
		
		/*	Maybe concatenate $content values from previous matching,
			and support $content being a string or an array. */
		if ( is_array( $content ) ) {
			$content = implode( ' ', $content );
		}
		
		/* Search for shortcodes. */
		$count = preg_match_all( "/$regex/", $content, $matches );
		
		/*	If we have shortcodes, extract their attributes and maybe recurse,
			otherwise return the shortcodes we've already found. */
		if ( $count ) {
			
			/*	Loop over each attribute string, extract attributes, and save them
				in an array keyed by the shortcode name. */
			foreach ( $matches[3] as $index => $attributes ) {
				
				/* Create a key for the shortcode name if we haven't already. */
				if ( empty( $existing[ $matches[2][ $index ] ] ) ) {
					$existing[ $matches[2][ $index ] ] = array();
				}
				
				/* Parse shortcode atts. */
				$shortcode_data = shortcode_parse_atts( $attributes );
				
				/* Save the found attributes under the key. */
				$existing[ $matches[2][ $index ] ][] = $shortcode_data;
			}
			
			/* Recurse as needed. */
			return $this->parseShortcodes( $regex, $matches[5], $existing );
			
		} else {
			return $existing;
		}
	
	}
	
	/**
	 * Fetch the referenced post content and then read the shortcodes inside it.
	 * The styles attached in the attributes should be saved inside 'theodore_page_style' post meta.
	 *
	 * @param int $post_id the ID of the post
	 */
	public function savePost ( $post_id, $post, $update ) {
		
		if ( 'auto-draft' === $post->post_status || 'revision' === $post->post_type ) {
			return;
		}
 
		$shortcode_regex = get_shortcode_regex();
		$shortcode_data = $this->parseShortcodes( $shortcode_regex, $post->post_content );
		
		foreach ( $shortcode_data as $k => $v ) {
			$name = str_replace( '_', ' ', $k );
			$name = ucwords( $name );
			$name = str_replace( ' ', '', $name );
			if ( method_exists( $this, 'applyTo' . $name ) && is_callable( array( $this, 'applyTo' . $name ) ) ) {
				foreach ( $v as $data ) {
					call_user_func( array( $this, 'applyTo' . $name ), $data );
				}
			}
		}
		
		update_post_meta( $post_id, 'theodore_shortcode_style', self::$css );
		
	}
	
	private function applyToRow ( $data ) {
		extract( shortcode_atts( array(
			'id' => '',
			'background_color' => '',
			'background_position' => '',
			'background_repeat' => '',
			'background_size' => '',
			'background_attachment' => '',
			'cover_background_color' => '',
			'cover_opacity' => '',
			'text_color' => '',
			'header_color' => '',
			'link_color' => '',
			'hovered_link_color' => '',
			'border_top_color' => '',
			'border_top_width' => '',
			'border_bottom_color' => '',
			'border_bottom_width' => ''
		), $data) );
		
		$selector = $id != '' ? '#' . str_replace( ' ', '', $id ) : '#custom-row-' . self::$newRowID++;
		
		/* Row element style */
		$rule = '';
		$rule.= $selector . '{';
		$rule.= $background_color != '' ? 'background-color: ' . esc_attr( $background_color ) . ';' : '';
		$rule.= $background_position != '' ? 'background-position: ' . esc_attr( $background_position ) . ';' : '';
		$rule.= $background_repeat != '' ? 'background-repeat: ' . esc_attr( $background_repeat ) . ';' : '';
		$rule.= $background_size != '' ? 'background-size: ' . esc_attr( $background_size ) . ';' : '';
		$rule.= $background_attachment != '' ? 'background-attachment: ' . esc_attr( $background_attachment ) . ';' : '';
		$rule.= $text_color != '' ? 'color: ' . esc_attr( $text_color ) . ';' : '';
		$rule.= $border_top_width != '' ? 'border-top-width: ' . esc_attr( $border_top_width ) . ';' : '';
		$rule.= $border_top_color != '' ? 'border-top-color: ' . esc_attr( $border_top_color ) . ';' : '';
		$rule.= $border_bottom_width != '' ? 'border-bottom-width: ' . esc_attr( $border_bottom_width ) . ';' : '';
		$rule.= $border_bottom_color != '' ? 'border-bottom-color: ' . esc_attr( $border_bottom_color ) . ';' : '';
		$rule.='}';
		
		/* Cover opacity */
		$rule.= $selector . ' .column:after {';
		
		$cover_opacity = floatval( $cover_opacity ) > 0 && floatval( $cover_opacity ) < 1 ? floatval( $cover_opacity ) : 0;
		$cover = $cover_background_color == 'white' ? 'rgba(255, 255, 255, ' . $cover_opacity . ')' : 'rgba(0, 0, 0, ' . $cover_opacity . ')';
		$rule.= 'background-color: ' . $cover . ';';
		
		$rule.= '}';
		
		/* Headers color */
		if ( esc_attr( $header_color ) != '' ) {
			$headers = array();
			for ( $i = 1; $i <= 6; $i++ ) {
				$headers[]= $selector . ' h' . $i;
			}
			$rule.= implode( ', ', $headers ) . '{color: ' . esc_attr( $header_color ) . ';}';
		}
		
		/* Link styles */
		$rule.= $link_color != '' ? $selector . ' a {color: ' . esc_attr( $link_color ) . ';}' : '';
		$rule.= $hovered_link_color != '' ? $selector . ' a:hover {color: ' . esc_attr( $hovered_link_color ) . ';}' : '';
		
		self::$css .= $rule;
	}
	
	private function applyToColumn ( $data ) {
		extract( shortcode_atts( array(
			'id' => '',
			'grid' => '',
			'background_color' => '',
			'background_position' => '',
			'background_repeat' => '',
			'background_size' => '',
			'background_attachment' => '',
			'cover_background_color' => '',
			'cover_opacity' => '',
			'text_color' => '',
			'header_color' => '',
			'link_color' => '',
			'hovered_link_color' => '',
			'border_right_color' => '',
			'border_right_width' => ''
		), $data) );
		
		$selector = $id != '' ? '#' . str_replace( ' ', '', $id ) : '#custom-column-' . self::$newColumnID++ . '.column';
		
		/* Column element style */
		$rule = '';
		$rule.= $selector . '{';
		$rule.= $background_color != '' ? 'background-color: ' . esc_attr( $background_color ) . ';' : '';
		$rule.= $background_position != '' ? 'background-position: ' . esc_attr( $background_position ) . ';' : '';
		$rule.= $background_repeat != '' ? 'background-repeat: ' . esc_attr( $background_repeat ) . ';' : '';
		$rule.= $background_size != '' ? 'background-size: ' . esc_attr( $background_size ) . ';' : '';
		$rule.= $background_attachment != '' ? 'background-attachment: ' . esc_attr( $background_attachment ) . ';' : '';
		$rule.= $text_color != '' ? 'color: ' . esc_attr( $text_color ) . ';' : '';
		$rule.= $border_right_width != '' ? 'border-right-width: ' . esc_attr( $border_right_width ) . ';' : '';
		$rule.= $border_right_color != '' ? 'border-right-color: ' . esc_attr( $border_right_color ) . ';' : '';
		$rule.='}';
		
		/* Cover opacity */
		if ( trim( $cover_opacity ) != '' ) {
			$rule.= $selector . ':after {';
			
			$cover_opacity = floatval( $cover_opacity ) > 0 && floatval( $cover_opacity ) < 1 ? floatval( $cover_opacity ) : 0;
			$cover = $cover_background_color == 'white' ? 'rgba(255, 255, 255, ' . $cover_opacity . ')' : 'rgba(0, 0, 0, ' . $cover_opacity . ')';
			$rule.= 'background-color: ' . $cover . ';';
			
			$rule.= '}';
		}
		
		/* Headers color */
		if ( esc_attr( $header_color ) != '' ) {
			$headers = array();
			for ( $i = 1; $i <= 6; $i++ ) {
				$headers[]= $selector . ' h' . $i;
			}
			$rule.= implode( ', ', $headers ) . '{color: ' . esc_attr( $header_color ) . ';}';
		}
		
		/* Link styles */
		$rule.= $link_color != '' ? $selector . ' a {color: ' . esc_attr( $link_color ) . ';}' : '';
		$rule.= $hovered_link_color != '' ? $selector . ' a:hover {color: ' . esc_attr( $hovered_link_color ) . ';}' : '';
		
		$grid = $grid == '' ? 12 : ( int ) $grid;
		
		self::$css .= $rule;
	}
	
	private function applyToSocialIconLinks ( $data ) {
		extract( shortcode_atts( array(
			'id' => '',
			'icon_color' => '',
			'icon_color_on_hover' => ''
		), $data) );
		
		$rule = '';
		$element_id = self::$newSocialIconLinksID++;
		
		$selector = '#social-icon-links-' . esc_attr( $element_id ) . ' a';
		$rule.= $selector . '{';
		$rule.= $icon_color != '' ? 'color: ' . esc_attr( $icon_color ) . ';' : '';
		$rule.='}';
		
		$selector = '#social-icon-links-' . esc_attr( $element_id ) . ' a:hover';
		$rule.= $selector . '{';
		$rule.= $icon_color_on_hover != '' ? 'color: ' . esc_attr( $icon_color_on_hover ) . ';' : '';
		$rule.='}';
		
		self::$css .= $rule;
	}
	
	public function applyToButton ( $data ) {
		
		extract(shortcode_atts( array(
			'text_color' => '',
			'background_color' => ''
		), $data));
		
		$rule = '';
		$element_id = self::$newButtonID++;
		
		$selector = '#module-button-' . esc_attr( $element_id );
		$rule.= $selector . '{';
		$rule.= $text_color != '' ? 'color: ' . esc_attr( $text_color ) . ';' : '';
		$rule.= $background_color != '' ? 'background-color: ' . esc_attr( $background_color ) . '; border-color: ' . esc_attr( $background_color ) . ';' : '';
		$rule.='}';
		
		self::$css .= $rule;
		
	}
	
}

?>